// FileMaker REST API for node.js

var express = require('express');
var router = express.Router({caseSensitive: true});
var errorHandler = require('./error');
var util = require('./util');


router.route('/')
.all(function(req, res, next){
	//block access beside Tableau connector
	if(!util.isFromTableau(req)){
		//return errorHandler.handleError('BadRequest', req, res, 'Invalid Service Request');
	}

	if(util.validateRequest(req, res)) {
		next();
	}
})

/**
 * @api {POST} /cursor/:solution/:layout/  createCursor
 * @apiDescription Create server cursor to do query.
 * @apiName createCursor
 * @apiGroup Cursor
 *
 * @apiUse AccessTokenHeader
 *
 * @apiUse FindParameters
 *
 * @apiSuccess {String} errorCode "0"
 * @apiSuccess {String} cursor token access token for subsequence cursor api calls.
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "errorCode": "0"
 *       "cursorToken": "[...]"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /cursor/:solution/:layout/
 */
.post(function(req, res, next){
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': res.locals.solution,
		'layout': res.locals.layout,
		'token': res.locals.token,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr
	};
	if(req.body['query']){
		params.query = req.body['query'];
	}
//	else{
//		return errorHandler.handleError('BadRequest', req, res, "Query parameter is missing.")
//	}
	if(req.body['sort']){
		params.sort = req.body['sort'];
	}
	try{
		util.thrift_client.createCursor(params,
	 	function(thrifError, thrifResult){
			return util.handleThrifReturn(thrifError, thrifResult, req, res);
	 	});
	} catch (err){
		util.thriftExceptionHandler(err, req, res);
	}
})


/**
 * @api {GET} /cursor/:solution/:layout[?range=:rangeOfRecords] fetch cursor
 * @apiDescription Get all or subset of records in specified cursor.
 * All records will be included in result set if no query parameters are given. <br>
 * Optional parameters "range" can be used to specify range in result set.<br>
 *
 * @apiName fetchCursor
 * @apiGroup Cursor
 *
 * @apiParamExample {json} Fetch with Range
 *	?range=50
 *
 * @apiUse AccessTokenHeader
 *
 * @apiSuccess {String} result "OK" or Draco Error Code
 * @apiSuccess {String} data JSON Array of row objects
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "records": "[...]"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /cursor/:solution/:layout?range=:rangeOfRecords
 */
.get(function(req, res, next){
	if(!res.locals.cursorToken){
		return errorHandler.handleError('BadRequest', req, res, 'Missing cursor token.')
	}
	 var ipStr = util.parseClientIp(req);
	 var params = {
		 'solution': res.locals.solution,
		 'layout': res.locals.layout,
		 'cursorToken': res.locals.cursorToken,
		 'token': res.locals.token,
		 'requestMethod': req.method,
		 'requestPath': req.originalUrl,
		 'requestIp': ipStr
	 };

	 if(req.query['_limit']){
		 params.limit = req.query['_limit'];
		 if(isNaN(params.limit) || params.limit<=0){
			 return errorHandler.handleError('BadRequest', req, res, 'The _limit must be an integer with value greater than 0.')
		 }
	 }
//	 if(! params.range){ //if range is missing, set it to 500 as default range
//		 params.range = '500'; //TODO:This limit should be enforce by backend in future
//	 }

	 try{
		 util.thrift_client.fetchCursor(params,
			 function(thrifError, thrifResult){
				return util.handleThrifReturn(thrifError, thrifResult, req, res);
		});
	 } catch (err){
		 util.thriftExceptionHandler(err, req, res);
	 }
})

/**
 * @api {delete} /cursor/:solution/:layout closeCursor
 * @apiDescription close server cursor in the session
 * @apiName closeCursor
 * @apiGroup Cursor
 *
 * @apiUse AccessTokenHeader
 *
 * @apiSuccess {String} result Logout result Status.
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK"
 *     }
 *
 *
 * @apiUse UnauthorizedError
 *
 * @apiSampleRequest /cursor/:solution/:layout
 */
.delete(function(req, res, next){
	if(!res.locals.cursorToken){
		return errorHandler.handleError('BadRequest', req, res, 'Missing cursor token.')
	}
	try{
		var ipStr = util.parseClientIp(req);
		util.thrift_client.closeCursor(
			{ 'solution': res.locals.solution,
				'layout': reqs.locals.layoutName,
				'cursorToken': res.locals.cursorToken,
				'token': res.locals.token,
				'requestMethod': req.method,
				'requestPath': req.originalUrl,
				'requestIp': ipStr
				},
			function(thrifError, thrifResult){
				return util.handleThrifReturn(thrifError, thrifResult, req, res);
		});
	} catch (err){
		util.thriftExceptionHandler(err, req, res);
	}
})

router.route('/reset')
.all(function(req, res, next){
	if(util.validateRequest(req, res)) {
		next();
	}
})
.post(function(req, res, next){
	if(!res.locals.cursorToken){
		return errorHandler.handleError('BadRequest', req, res, 'Missing cursor token.')
	}
	var ipStr = util.parseClientIp(req);
	var params = {
			 'solution': res.locals.solution,
			 'layout': res.locals.layout,
			 'cursorToken': res.locals.cursorToken,
			 'token': res.locals.token,
			 'requestMethod': req.method,
			 'requestPath': req.originalUrl,
			 'requestIp': ipStr
	};
	//reset the cursor to start at a record right after the given recordId
	if(req.body['recordId']){
		params.id = req.body['recordId'];
	}

	try{
		util.thrift_client.resetCursor(params,
	 	function(thrifError, thrifResult){
			return util.handleThrifReturn(thrifError, thrifResult, req, res);
	 	});
	} catch (err){
		util.thriftExceptionHandler(err, req, res);
	}
})
module.exports.router = router;
